<?php

/**
 * Calls the class on the post edit screen.
 */
function call_LaniakeaMetaBoxes()
{
    new LaniakeaMetaBoxes();
}

if (is_admin()) {
    add_action('load-post.php',     'call_LaniakeaMetaBoxes');
    add_action('load-post-new.php', 'call_LaniakeaMetaBoxes');
}

/**
 * The Class.
 */
class LaniakeaMetaBoxes
{

    /**
     * Hook into the appropriate actions when the class is constructed.
     */
    public function __construct()
    {
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('save_post',      array($this, 'save'));
    }

    /**
     * Adds the meta box container.
     */
    public function add_meta_box($post_type)
    {
        // Limit meta box to certain post types.
        $post_types = array('post', 'page');

        if (in_array($post_type, $post_types)) {

            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');

            add_meta_box(
                'laniakea_header_box',
                __('Laniakea | Intestazione Pagina', 'laniakea'),
                array($this, 'render_header_box'),
                $post_type,
                'advanced',
                'high'
            );
        }
    }

    /**
     * Save the meta when the post is saved.
     *
     * @param int $post_id The ID of the post being saved.
     */
    public function save($post_id)
    {

        /*
		 * We need to verify this came from the our screen and with proper authorization,
		 * because save_post can be triggered at other times.
		 */

        // Check if our nonce is set.
        if (!isset($_POST['laniakea_box_nonce'])) {
            return $post_id;
        }

        $nonce = $_POST['laniakea_box_nonce'];

        // Verify that the nonce is valid.
        if (!wp_verify_nonce($nonce, 'laniakea_box')) {
            return $post_id;
        }

        /*
		 * If this is an autosave, our form has not been submitted,
		 * so we don't want to do anything.
		 */
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return $post_id;
        }

        // Check the user's permissions.
        if ('page' == $_POST['post_type']) {
            if (!current_user_can('edit_page', $post_id)) {
                return $post_id;
            }
        } else {
            if (!current_user_can('edit_post', $post_id)) {
                return $post_id;
            }
        }

        /* OK, it's safe for us to save the data now. */

        // Sanitize the user input.
        $header_title = sanitize_text_field($_POST['laniakea_header_title']);
        $header_show = isset($_POST['laniakea_show_header']) ? 1 : 0;
        $transparent_header = isset($_POST['laniakea_transparent_header']) ? 1 : 0;

        // Update the meta field.
        update_post_meta($post_id, 'laniakea_header_title', $header_title);
        update_post_meta($post_id, 'laniakea_show_header', $header_show);
        update_post_meta($post_id, 'laniakea_transparent_header', $transparent_header);
        update_post_meta($post_id, 'laniakea_header_bg', sanitize_hex_color($_POST['laniakea_header_bg']));
    }


    /**
     * Render Meta Box content.
     *
     * @param WP_Post $post The post object.
     */
    public function render_header_box($post)
    {

        // Add an nonce field so we can check for it later.
        wp_nonce_field('laniakea_box', 'laniakea_box_nonce');

        // Use get_post_meta to retrieve an existing value from the database.
        $laniakea_show_header = get_post_meta($post->ID, 'laniakea_show_header', true);
        $laniakea_header_title = get_post_meta($post->ID, 'laniakea_header_title', true);
        $laniakea_transparent_header =  get_post_meta($post->ID, 'laniakea_transparent_header', true);
        $laniakea_header_bg = get_post_meta($post->ID, 'laniakea_header_bg', true);

        // Display the form, using the current value.
?>
        <div class="laniakea-meta-box">

            <div class="field-group">
                <input type="checkbox" id="laniakea_show_header" name="laniakea_show_header" value="1" <?php checked($laniakea_show_header, 1, true); ?> />
                <label for="laniakea_show_header">
                    <?php _e('Mostra Intestazione', 'laniakea'); ?>
                </label>
            </div>

            <div class="field-group">
                <input type="checkbox" id="laniakea_transparent_header" name="laniakea_transparent_header" value="1" <?php checked($laniakea_transparent_header, 1, true); ?> />
                <label for="laniakea_transparent_header">
                    <?php _e('Menu trasparente', 'laniakea'); ?>
                </label>
            </div>

            <div class="field-group color-picker">
                <label for="laniakea_header_bg">
                    <?php _e('Background Intestazione', 'laniakea'); ?>
                </label>
                <input type="text" id="laniakea_header_bg" name="laniakea_header_bg" value="<?php echo esc_attr($laniakea_header_bg); ?>" />
            </div>

            <div class="field-group">
                <label for="laniakea_header_title">
                    <?php _e('Titolo Intestazione', 'laniakea'); ?>
                </label>
                <input type="text" id="laniakea_header_title" name="laniakea_header_title" value="<?php echo esc_attr($laniakea_header_title); ?>" style="width:100%" />
            </div>
        </div>
<?php
    }
}
